<?php

namespace App\Http\Controllers;

use App\Models\SystemSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class RedirectController extends Controller
{
    /**
     * Redirect user to GoHighLevel form with their contact ID
     */
    public function redirectToGhl()
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $user = Auth::user();
        
        // Si es admin, redirigir al dashboard de admin
        if ($user->isAdmin()) {
            return redirect()->route('admin.dashboard');
        }
        
        if (!$user->isActive()) {
            Auth::logout();
            return redirect()->route('login')
                ->with('error', 'Tu cuenta está desactivada. Contacta al administrador.');
        }

        $ghlFormUrl = SystemSetting::getValue('ghl_form_url');
        
        if (!$ghlFormUrl) {
            Log::error('GHL form URL not configured', [
                'user_id' => $user->id,
            ]);
            Auth::logout();
            return redirect()->route('login')
                ->with('error', 'El sistema no está configurado correctamente. Contacta al administrador.');
        }

        if (!$user->ghl_contact_id) {
            Log::warning('User without GHL contact ID attempted redirect', [
                'user_id' => $user->id,
                'email' => $user->email,
            ]);
            Auth::logout();
            return redirect()->route('login')
                ->with('error', 'Tu cuenta no tiene un ID de contacto configurado. Contacta al administrador.');
        }

        // Build URL with GHL contact ID as parameter
        $separator = strpos($ghlFormUrl, '?') !== false ? '&' : '?';
        $redirectUrl = $ghlFormUrl . $separator . 'sp=' . urlencode($user->ghl_contact_id);

        Log::info('User redirected to GHL form', [
            'user_id' => $user->id,
            'email' => $user->email,
            'ghl_contact_id' => $user->ghl_contact_id,
            'redirect_url' => $redirectUrl,
        ]);

        return redirect($redirectUrl);
    }
}
